using System;
using System.Data;
using System.Collections;
using System.Text.RegularExpressions;
using gov.va.med.vbecs.DAL.HL7AL;
using TABLE = gov.va.med.vbecs.Common.VbecsTables;
using ARTIFICIAL = gov.va.med.vbecs.Common.DatabaseConstants.ArtificialColumnNames;
namespace gov.va.med.vbecs.BOL
{
	/// <summary>
	/// Summary description for OrderedComponent.
	/// </summary>
	public class OrderedComponent : PatientOrder, IRecordStatusCode
	{
		// CR3114: Cache the specimen object; GetSpecimen is called so many times, this is the only efficient way to do things.
		private BOL.Specimen _specimen;


		private Guid _orderedComponentGUID;
		private Common.TaskStatus _taskStatusCode;
		private string _cprsOrderNumber;
		private string _orderSpecialInstructions;
		private BloodComponentClass _componentOrdered;
		decimal _quantity;
		DateTime _requiredDateTime;
		private string _orderUrgencyCode;
		private string _orderUrgencyText;
		private Common.OrderStatus _orderStatusCode;
		private string _orderStatusText;
		private Guid _patientSpecimenGuid; // CR3114
		private Common.SpecimenStatus _specimenStatusCode;
		private string _specimenStatusText;
		private string _VBECSOrderNumber;
		private bool _preOpIndicator;
		private bool _inappropriateRequestIndicator;
		private Guid _cancellationCommentGuid;
		private string _orderedComponentComments;
		private string _labOrderNumber;
		private string _labTestID;
		private string _specimenAccessionNumber;
		private Common.RecordStatusCode _recordStatusCode;
		private bool _specimenRequiredIndicator;
		private string _noInappropriateReportText;
		private bool _emergencyOrderIndicator;
		private byte[] _rowVersion;
		private const string LAB_TEST_TABLE_NAME = "LabTest";


		
		#region constructors


		///<Developers>
		///	<Developer>Rob Heiberger</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/18/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="6386"> 
		///		<ExpectedInput>Valid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="6387"> 
		///		<ExpectedInput>Invalid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Constructor
		/// </summary>
		/// <param name="dtRow"></param>
		public OrderedComponent(DataRow dtRow) : base(dtRow)
		{
			LoadFromDataRow(dtRow);
		}

		#endregion


		#region properties

		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/10/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="7505"> 
		///		<ExpectedInput>Valid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="7506"> 
		///		<ExpectedInput>Invalid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// NoInappropriateReportText
		/// </summary>
		public string NoInappropriateReportText
		{
			get
			{
				return _noInappropriateReportText;
			}
			set
			{
				_noInappropriateReportText = value;
			}
		}

		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/11/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="7509"> 
		///		<ExpectedInput>Valid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="7510"> 
		///		<ExpectedInput>Invalid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// The order could have been cancelled in CPRS before after the screen brought up, but
		/// prior to the save.  This is a check to ensure the order is still pending.
		/// </summary>
		public bool StillActive
		{
			get
			{
				if (this.OrderedComponentGuid != Guid.Empty)
				{
					Common.OrderStatus status = Common.Utility.GetOrderStatusCodeFromString(DAL.OrderedComponent.GetOrderedComponent(this.OrderedComponentGuid).Rows[0][TABLE.OrderedTest.OrderStatusCode].ToString());
					return status != Common.OrderStatus.Cancelled;
				}
				else
				{
					return false;
				}
			}
		}

		///<Developers>
		///	<Developer>Rob Heiberger</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/18/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="6344"> 
		///		<ExpectedInput>Valid String</ExpectedInput>
		///		<ExpectedOutput>Valid String</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="6345"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Specimen accession number from VistA
		/// </summary>
		public string SpecimenAccessionNumber
		{
			get
			{
				return this._specimenAccessionNumber;
			}
			set
			{
				this._specimenAccessionNumber = value;
			}
		}

		///<Developers>
		///	<Developer>Rob Heiberger</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/18/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="6346"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>boolean</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="6347"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Paritally Implements BR_16.06
		/// </summary>
		public bool SpecimenRequiredIndicator
		{
			get
			{
				return this._specimenRequiredIndicator;
			}
		}

		///<Developers>
		///	<Developer>Krzysztof Dobranowski</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>4/29/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="2720"> 
		///		<ExpectedInput>Valid Guid</ExpectedInput>
		///		<ExpectedOutput>Valid Guid</ExpectedOutput>
		///	</Case>
		///	
		///<Case type="1" testid ="387"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// OrderedComponentGuid
		/// </summary>
		public Guid OrderedComponentGuid
		{
			get
			{
				return this._orderedComponentGUID;
			}

			set
			{
				this._orderedComponentGUID = value;
			}
		}

		///<Developers>
		///	<Developer>Krzysztof Dobranowski</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>4/29/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="2721"> 
		///		<ExpectedInput>String containing the valid task status code</ExpectedInput>
		///		<ExpectedOutput>String containing the valid task status code</ExpectedOutput>
		///	</Case>
		///	
		///<Case type="1" testid ="389"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// TaskStatusCode
		/// </summary>
		public Common.TaskStatus TaskStatusCode
		{
			get
			{
				return this._taskStatusCode;
			}

			set
			{
				this._taskStatusCode = value;
			}
		}

		///<Developers>
		///	<Developer>Rob Heiberger</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/18/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="6348"> 
		///		<ExpectedInput>Valid integer</ExpectedInput>
		///		<ExpectedOutput>Valid integer</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="6349"> 
		///		<ExpectedInput>Invalid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// The status of the record
		/// </summary>
		public Common.RecordStatusCode RecordStatusCode
		{
			get
			{
				return _recordStatusCode;
			}
			set
			{
				_recordStatusCode = value;
			}
		}

		///<Developers>
		///	<Developer>Krzysztof Dobranowski</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>4/29/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="2722"> 
		///		<ExpectedInput>Valid BloodComponentClass object</ExpectedInput>
		///		<ExpectedOutput>Valid BloodComponentClass object</ExpectedOutput>
		///	</Case>
		///	
		///<Case type="1" testid ="390"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// ComponentOrdered
		/// </summary>
		public BloodComponentClass ComponentOrdered
		{
			get
			{
				return this._componentOrdered;
			}

			set
			{
				this._componentOrdered = value;
			}
		}

		///<Developers>
		///	<Developer>Krzysztof Dobranowski</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>4/29/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="2723"> 
		///		<ExpectedInput>Integer value</ExpectedInput>
		///		<ExpectedOutput>Integer value</ExpectedOutput>
		///	</Case>
		///	
		///<Case type="1" testid ="393"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Quantity
		/// </summary>
		public decimal Quantity
		{
			get
			{
				return this._quantity;
			}

			set
			{
				this._quantity = value;
			}
		}

		
		///<Developers>
		///	<Developer>Rob Heiberger</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/18/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="6350"> 
		///		<ExpectedInput>Valid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="6351"> 
		///		<ExpectedInput>Invalid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// CPRSOrderNumber
		/// </summary>
		public string CPRSOrderNumber
		{
			get
			{
				return this._cprsOrderNumber;
			}

			set
			{
				Regex regEx = new Regex(@"[\w{20}]");
				if (!regEx.IsMatch(value))
				{
					throw new Exception("Value must be between 1-20 alphanumeric characters");
				}
				this._cprsOrderNumber = value;
			}
		}

		///<Developers>
		///	<Developer>Rob Heiberger</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/18/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="6352"> 
		///		<ExpectedInput>Valid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="6353"> 
		///		<ExpectedInput>Invalid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// RequiredDateTime
		/// </summary>
		public DateTime RequiredDateTime
		{
			get
			{
				return this._requiredDateTime;
			}

			set
			{
				this._requiredDateTime = value;
			}
		}

		///<Developers>
		///	<Developer>Rob Heiberger</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/18/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="6354"> 
		///		<ExpectedInput>Valid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="6355"> 
		///		<ExpectedInput>Invalid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///	
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// OrderSpecialInstructions
		/// </summary>
		public String OrderSpecialInstructions
		{
			get
			{
				return this._orderSpecialInstructions;
			}

			set
			{
				this._orderSpecialInstructions = value;
			}
		}

		///<Developers>
		///	<Developer>Rob Heiberger</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/18/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="6356"> 
		///		<ExpectedInput>Valid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="6357"> 
		///		<ExpectedInput>Invalid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// OrderSpecialInstructions
		/// </summary>
		public String OrderedComponentComments
		{
			get
			{
				return this._orderedComponentComments;
			}

			set
			{
				this._orderedComponentComments = value;
			}
		}

		///<Developers>
		///	<Developer>Rob Heiberger</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/18/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="6358"> 
		///		<ExpectedInput>Valid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="6359"> 
		///		<ExpectedInput>Invalid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// OrderUrgencyCode
		/// </summary>
		public string OrderUrgencyCode
		{
			get
			{
				return this._orderUrgencyCode;
			}

			set
			{
				this._orderUrgencyCode = value;
			}
		}

		///<Developers>
		///	<Developer>Rob Heiberger</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/18/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="6360"> 
		///		<ExpectedInput>Valid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="6361"> 
		///		<ExpectedInput>Invalid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// OrderUrgencyText
		/// </summary>
		public string OrderUrgencyText
		{
			get
			{
				return this._orderUrgencyText;
			}

			set
			{
				this._orderUrgencyText = value;
			}
		}

		///<Developers>
		///	<Developer>Rob Heiberger</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/18/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="6362"> 
		///		<ExpectedInput>Valid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="6363"> 
		///		<ExpectedInput>Invalid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		///OrderStatusCode
		/// </summary>
		public Common.OrderStatus OrderStatusCode
		{
			get
			{
				return this._orderStatusCode;
			}

			set
			{
				this._orderStatusCode = value;
			}
		}

		///<Developers>
		///	<Developer>Rob Heiberger</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/18/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="6364"> 
		///		<ExpectedInput>Valid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="6365"> 
		///		<ExpectedInput>Invalid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// OrderStatusText
		/// </summary>
		public string OrderStatusText
		{
			get
			{
				return this._orderStatusText;
			}

			set
			{
				this._orderStatusText = value;
			}
		}

		///<Developers>
		///	<Developer>C. Jensen</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/18/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="6366"> 
		///		<ExpectedInput>Valid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="6367"> 
		///		<ExpectedInput>Invalid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// CR3114: 3114todo: unit tests and event on previous property (below)
		/// This indicates whether the component has a specimen association.
		/// </summary>
		public bool SpecimenIdentified
		{
			get
			{
				return _patientSpecimenGuid != Guid.Empty;
			}
		}


		///<Developers>
		///	<Developer>Rob Heiberger</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/18/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="6368"> 
		///		<ExpectedInput>Valid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="6369"> 
		///		<ExpectedInput>Invalid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// SpecimenStatusCode
		/// </summary>
		public Common.SpecimenStatus SpecimenStatusCode
		{
			get
			{
				return this._specimenStatusCode;
			}

			set
			{
				this._specimenStatusCode = value;
			}
		}

		///<Developers>
		///	<Developer>Rob Heiberger</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/18/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="6370"> 
		///		<ExpectedInput>Valid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="6371"> 
		///		<ExpectedInput>Invalid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// SpecimenStatusText
		/// </summary>
		public string SpecimenStatusText
		{
			get
			{
				return this._specimenStatusText;
			}

			set
			{
				this._specimenStatusText = value;
			}
		}

		///<Developers>
		///	<Developer>Rob Heiberger</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/18/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="6372"> 
		///		<ExpectedInput>Valid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///	
		///<Case type="1" testid ="6373"> 
		///		<ExpectedInput>Invalid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// VBECSOrderNumber
		/// </summary>
		public string VBECSOrderNumber
		{
			get
			{
				return this._VBECSOrderNumber;
			}

			set
			{
				this._VBECSOrderNumber = value;
			}
		}

  		///<Developers>
		///	<Developer>Rob Heiberger</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/18/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="6374"> 
		///		<ExpectedInput>Valid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="6375"> 
		///		<ExpectedInput>Invalid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// PreOpIndicator
		/// </summary>
		public bool PreOpIndicator
		{
			get
			{
				return this._preOpIndicator;
			}

			set
			{
				this._preOpIndicator = value;
			}
		}

		///<Developers>
		///	<Developer>Rob Heiberger</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/18/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="6376"> 
		///		<ExpectedInput>Valid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="6377"> 
		///		<ExpectedInput>Invalid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// InappropriateRequestIndicator
		/// </summary>
		public bool InappropriateRequestIndicator
		{
			get
			{
				return this._inappropriateRequestIndicator;
			}

			set
			{
				this._inappropriateRequestIndicator = value;
			}
		}


		///<Developers>
		///	<Developer>Rob Heiberger</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/18/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="6378"> 
		///		<ExpectedInput>Valid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="6379"> 
		///		<ExpectedInput>Invalid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// CancellationCommentGUID
		/// </summary>
		public Guid CancellationCommentGUID
		{
			get
			{
				return this._cancellationCommentGuid;
			}
			set
			{
				this._cancellationCommentGuid = value;
			}
		}

		///<Developers>
		///	<Developer>Rob Heiberger</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/18/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="6380"> 
		///		<ExpectedInput>Valid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="6381"> 
		///		<ExpectedInput>Invalid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Lab Order Number - used to validate specimen with VistA
		/// </summary>
		public string LabOrderNumber
		{
			get
			{
				return _labOrderNumber;
			}
		}

		///<Developers>
		///	<Developer>Rob Heiberger</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/18/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="6382"> 
		///		<ExpectedInput>Valid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="6383"> 
		///		<ExpectedInput>Invalid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Lab Test ID (from VistA)
		/// </summary>
		public string LabTestId
		{
			get
			{
				return _labTestID;
			}
			set
			{
				_labTestID = value;
			}
		}

		///<Developers>
		///	<Developer>C. Jensen</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>5/13/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="7299"> 
		///		<ExpectedInput>Valid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="7300"> 
		///		<ExpectedInput>Invalid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// EmergencyOrderIndicator
		/// </summary>
		public bool EmergencyOrderIndicator
		{
			get
			{	
				return _emergencyOrderIndicator;
			}
			set
			{
				_emergencyOrderIndicator = value;
			}
		}

		///<Developers>
		///	<Developer>Rob Heiberger</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/18/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="6384"> 
		///		<ExpectedInput>Valid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="6385"> 
		///		<ExpectedInput>Invalid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// RowVersion
		/// </summary>
		public new System.Byte[] RowVersion
		{
			get
			{
				return this._rowVersion;
			}
			set
			{
				if (value == null || value.Length > 8)
				{
					throw new ArgumentException("Invalid RowVersion argument", "value");
				}
				this._rowVersion = value;
			} 

		}

		#endregion


		#region methods


		/// <summary>
		/// CR3114
		/// Returns the specimen that is associated with this component
		/// </summary>
		/// <returns></returns>
		public BOL.Specimen GetSpecimen()
		{
			if (_specimen == null)
			{
				_specimen = new Specimen(_patientSpecimenGuid);
			}
			return _specimen;
		}


		/// <summary>
		/// CR3114
		/// Call this to associate a specimen with the component.
		/// </summary>
		/// <param name="specimen"></param>
		public void AssociateWithSpecimen(BOL.Specimen specimen)
		{
			_specimen = specimen;
			// Must do this because this property is what is checked for specimen associations.
            //CR 3554
		    if (specimen == null)
		    {
		        _patientSpecimenGuid = Guid.Empty;
		    }
		    else
		    {
		        _patientSpecimenGuid = specimen.SpecimenGuid;
		    }
		}


		///<Developers>
		///	<Developer>Krzysztof Dobranowski</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>4/29/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="2724"> 
		///		<ExpectedInput>Valid division code</ExpectedInput>
		///		<ExpectedOutput>The DataTable containing the list of component tasks</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="2725"> 
		///		<ExpectedInput>Invalid division code</ExpectedInput>
		///		<ExpectedOutput>The DataTable containing the list of component tasks</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Get the list of component tasks
		/// </summary>
		/// <returns></returns>
		public static DataTable GetComponentTaskList(string orderGroupNumber)
		{
			return DAL.OrderedComponent.GetComponentTaskList(orderGroupNumber);
		}

		///<Developers>
		///	<Developer>Rob Heiberger</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/18/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="6388"> 
		///		<ExpectedInput>Existing OrderedComponentGuid</ExpectedInput>
		///		<ExpectedOutput>DataTable</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="6389"> 
		///		<ExpectedInput>Non-Existent OrderedComponentGuid</ExpectedInput>
		///		<ExpectedOutput>Empty Table</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// GetOrderedComponent
		/// </summary>
		/// <param name="orderedComponentGuid"></param>
		/// <returns></returns>
		public static DataTable GetOrderedComponent(Guid orderedComponentGuid)
		{
			return DAL.OrderedComponent.GetOrderedComponent(orderedComponentGuid);
		}

		///<Developers>
		///	<Developer>Rob Heiberger</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/18/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="6390"> 
		///		<ExpectedInput>Valid lab test number and lab test id.</ExpectedInput>
		///		<ExpectedOutput>String</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="6391"> 
		///		<ExpectedInput>Empty strings</ExpectedInput>
		///		<ExpectedOutput>Empty string</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// GetAccessionByLabOrderNumber
		/// </summary>
		/// <param name="labOrderNumber"></param>
		/// <param name="labTestId"></param>
		/// <returns></returns>
		public static string GetAccessionByLabOrderNumber(string labOrderNumber, string labTestId)
		{
			if (labOrderNumber.Trim().Length == 0 || labTestId.Trim().Length == 0) return string.Empty;

			DataSet dsAccession = DAL.VAL.LabOrder.GetLabAccessionByLabOrderNumber(labOrderNumber);

				// Fixed problem associated with CR 1482

			foreach( DataTable dt in dsAccession.Tables )
			{
				if( dt.TableName == LAB_TEST_TABLE_NAME )
				{
					foreach (DataRow dr in dt.Rows)
					{
						if( dr.Table.Columns.Contains(TABLE.OrderedComponent.LabOrderNumber) && !dr.IsNull(TABLE.OrderedComponent.LabOrderNumber) 
							&& dr.Table.Columns.Contains(TABLE.OrderedComponent.LabTestId) && !dr.IsNull(TABLE.OrderedComponent.LabTestId)
							&& dr.Table.Columns.Contains(TABLE.OrderedComponent.SpecimenAccessionNumber) && !dr.IsNull(TABLE.OrderedComponent.SpecimenAccessionNumber)
							&& labOrderNumber == dr[TABLE.OrderedComponent.LabOrderNumber].ToString().Trim()
							&& labTestId == dr[TABLE.OrderedComponent.LabTestId].ToString().Trim() )
						{
								return dr[TABLE.OrderedComponent.SpecimenAccessionNumber].ToString();
						}
					}
				}
			}
			// If this point is reached in the code, the order was not accessioned.
			return string.Empty;

		}
		
		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>11/29/2004</CreationDate>
		///<TestCases>
		///		
		///<Case type="0" testid ="6392"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>boolean</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="7511"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Returns a boolean that says if an Ordered Component is Pending
		/// </summary>
		/// <returns></returns>
		public static bool IsOrderedComponentPending()
		{
			return DAL.OrderedComponent.IsOrderedComponentPending(int.MinValue);
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>11/29/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="6393"> 
		///		<ExpectedInput>Valid Parameters</ExpectedInput>
		///		<ExpectedOutput>boolean</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="6394"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Returns a boolean that says if an Ordered Component is Pending
		/// </summary>
		/// <param name="componentClassID"></param>
		/// <returns></returns>
		public static bool IsOrderedComponentPending(Common.ComponentClass componentClassID)
		{
			return DAL.OrderedComponent.IsOrderedComponentPending((int)componentClassID);
		}

		/// <summary>
		/// LoadFromDataRow
		/// </summary>
		/// <param name="dtRow"></param>
		protected new void LoadFromDataRow(DataRow dtRow)
		{
			if (dtRow.Table.Columns.Contains(TABLE.OrderedComponent.SpecimenAccessionNumber))
			{
				this.SpecimenAccessionNumber = dtRow.IsNull(TABLE.OrderedComponent.SpecimenAccessionNumber) ? String.Empty : dtRow[TABLE.OrderedComponent.SpecimenAccessionNumber].ToString();
			}
			if (dtRow.Table.Columns.Contains(TABLE.OrderedComponent.VbecsOrderNumber))
			{
				this.VBECSOrderNumber = dtRow.IsNull(TABLE.OrderedComponent.VbecsOrderNumber) ? String.Empty : dtRow[TABLE.OrderedComponent.VbecsOrderNumber].ToString();
			}
			if (dtRow.Table.Columns.Contains(TABLE.OrderedComponent.OrderedComponentGuid))
			{
				this.OrderedComponentGuid = dtRow.IsNull(TABLE.OrderedComponent.OrderedComponentGuid) ? Guid.NewGuid() : (System.Guid) dtRow[TABLE.OrderedComponent.OrderedComponentGuid];
			}
			if (dtRow.Table.Columns.Contains(TABLE.OrderedComponent.CprsOrderNumber))
			{
				this.CPRSOrderNumber = dtRow.IsNull(TABLE.OrderedComponent.CprsOrderNumber) ? String.Empty : dtRow[TABLE.OrderedComponent.CprsOrderNumber].ToString();
			}
			if (dtRow.Table.Columns.Contains(TABLE.OrderedComponent.RequiredUnitQuantity))
			{
				this.Quantity = dtRow.IsNull(TABLE.OrderedComponent.RequiredUnitQuantity) ? decimal.MinValue : (decimal) dtRow[TABLE.OrderedComponent.RequiredUnitQuantity];
			}
			if (dtRow.Table.Columns.Contains(TABLE.OrderedComponent.OrderUrgencyCode))
			{
				this.OrderUrgencyCode = dtRow.IsNull(TABLE.OrderedComponent.OrderUrgencyCode) ? String.Empty : dtRow[TABLE.OrderedComponent.OrderUrgencyCode].ToString();
			}
			if (dtRow.Table.Columns.Contains(TABLE.OrderUrgency.OrderUrgencyText))
			{
				this.OrderUrgencyText = dtRow.IsNull(TABLE.OrderUrgency.OrderUrgencyText) ? String.Empty : dtRow[TABLE.OrderUrgency.OrderUrgencyText].ToString();
			}
			if (dtRow.Table.Columns.Contains(TABLE.OrderedComponent.OrderedComponentComments))
			{
				this.OrderedComponentComments = dtRow.IsNull(TABLE.OrderedComponent.OrderedComponentComments) ? String.Empty : dtRow[TABLE.OrderedComponent.OrderedComponentComments].ToString();
			}
			if (dtRow.Table.Columns.Contains(TABLE.OrderedComponent.RequiredDatetime))
			{
				this.RequiredDateTime = dtRow.IsNull(TABLE.OrderedComponent.RequiredDatetime) ? DateTime.MinValue : (DateTime) dtRow[TABLE.OrderedComponent.RequiredDatetime];
			}
			if (dtRow.Table.Columns.Contains(TABLE.OrderedComponent.OrderStatusCode))
			{
				this.OrderStatusCode = dtRow.IsNull(TABLE.OrderedComponent.OrderStatusCode) ? Common.OrderStatus.Unknown : Common.Utility.GetOrderStatusCodeFromString(dtRow[TABLE.OrderedComponent.OrderStatusCode].ToString());
			}
			if (dtRow.Table.Columns.Contains(TABLE.OrderStatus.OrderStatusText))
			{
				this.OrderStatusText = dtRow.IsNull(TABLE.OrderStatus.OrderStatusText) ? String.Empty : dtRow[TABLE.OrderStatus.OrderStatusText].ToString();
			}
			if (dtRow.Table.Columns.Contains(TABLE.OrderedComponent.OrderSpecialInstructions))
			{
				this.OrderSpecialInstructions = dtRow.IsNull(TABLE.OrderedComponent.OrderSpecialInstructions) ? String.Empty : dtRow[TABLE.OrderedComponent.OrderSpecialInstructions].ToString();
			}
			if (dtRow.Table.Columns.Contains(TABLE.BloodUnit.RecordStatusCode))
			{
				this.RecordStatusCode =	dtRow.IsNull(TABLE.BloodUnit.RecordStatusCode) ? Common.RecordStatusCode.Unknown : Common.Utility.GetRecordStatusCodeFromString(dtRow[TABLE.BloodUnit.RecordStatusCode].ToString());
			}
			if (dtRow.Table.Columns.Contains(TABLE.ComponentClass.ComponentClassId))
			{
				DataTable dt = DAL.BloodComponentClass.GetComponentClasses();
				DataRow [] dr = dt.Select("ComponentClassId = " + (int) dtRow[TABLE.ComponentClass.ComponentClassId]); 
				this.ComponentOrdered = new BOL.BloodComponentClass(dr[0]);
			}
			if (dtRow.Table.Columns.Contains(TABLE.OrderedComponent.SpecimenRequiredIndicator))
			{
				this._specimenRequiredIndicator = dtRow.IsNull(TABLE.OrderedComponent.SpecimenRequiredIndicator) ? false : (bool) dtRow[TABLE.OrderedComponent.SpecimenRequiredIndicator];
			}
			if (dtRow.Table.Columns.Contains(TABLE.OrderedComponent.SpecimenStatusCode))
			{
				this.SpecimenStatusCode = dtRow.IsNull(TABLE.OrderedComponent.SpecimenStatusCode) ? Common.SpecimenStatus.Unknown : Common.Utility.GetSpecimenStatusCodeFromString(dtRow[TABLE.OrderedComponent.SpecimenStatusCode].ToString());
			}
			if (dtRow.Table.Columns.Contains(TABLE.SpecimenStatus.SpecimenStatusText))
			{
				this.SpecimenStatusText = dtRow.IsNull(TABLE.SpecimenStatus.SpecimenStatusText) ? String.Empty : dtRow[TABLE.SpecimenStatus.SpecimenStatusText].ToString();
			}
			if (dtRow.Table.Columns.Contains(TABLE.OrderedComponent.PreOpIndicator))
			{
				this.PreOpIndicator = dtRow.IsNull(TABLE.OrderedComponent.PreOpIndicator) ? false : (bool) dtRow[TABLE.OrderedComponent.PreOpIndicator];
			}
			if (dtRow.Table.Columns.Contains(TABLE.OrderedComponent.PatientSpecimenGuid))
			{
				this._patientSpecimenGuid = dtRow.IsNull(TABLE.OrderedComponent.PatientSpecimenGuid) ? Guid.Empty : (Guid) dtRow[TABLE.OrderedComponent.PatientSpecimenGuid];  // CR3114
			}
			if (dtRow.Table.Columns.Contains(TABLE.Patient.PatientGuid))
			{
				this.Patient = dtRow.IsNull(TABLE.Patient.PatientGuid) ? new Patient() : new Patient( (Guid) dtRow[TABLE.Patient.PatientGuid] );
			}
			if (dtRow.Table.Columns.Contains(TABLE.OrderedComponent.TaskStatusCode))
			{
				this.TaskStatusCode = dtRow.IsNull(TABLE.OrderedComponent.TaskStatusCode) ? Common.TaskStatus.Unknown : Common.Utility.GetTaskStatusCodeFromString(dtRow[TABLE.OrderedComponent.TaskStatusCode].ToString());
			}
			if( dtRow.Table.Columns.Contains(TABLE.OrderedComponent.LabOrderNumber))
			{
				if (!dtRow.IsNull(TABLE.OrderedComponent.LabOrderNumber))
				{
					_labOrderNumber = dtRow[TABLE.OrderedComponent.LabOrderNumber].ToString();
				}
			}
			if (dtRow.Table.Columns.Contains(TABLE.OrderedComponent.LabTestId))
			{
				this.LabTestId = dtRow.IsNull(TABLE.OrderedComponent.LabTestId) ? string.Empty : dtRow[TABLE.OrderedComponent.LabTestId].ToString();
			}
			if (dtRow.Table.Columns.Contains(TABLE.OrderedComponent.InappropriateRequestIndicator))
			{
				this.InappropriateRequestIndicator = dtRow.IsNull(TABLE.OrderedComponent.InappropriateRequestIndicator) ? false : (bool) dtRow[TABLE.OrderedComponent.InappropriateRequestIndicator];
			}
			if(dtRow.Table.Columns.Contains(TABLE.OrderedComponent.RowVersion))
			{
				if (!dtRow.IsNull(TABLE.OrderedComponent.RowVersion))
				{
					this.RowVersion = (byte[]) dtRow[TABLE.OrderedComponent.RowVersion];
				}
			}
		}


		///<Developers>
		///	<Developer>Carl Jensen</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>6/16/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="2928"> 
		///		<ExpectedInput>Valid division code</ExpectedInput>
		///		<ExpectedOutput>DataTable of component Orders</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="2929"> 
		///		<ExpectedInput>Invalid division code</ExpectedInput>
		///		<ExpectedOutput>Empty DataTable</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Returns a data table of Component Orders for a division
		/// </summary>
		/// <returns></returns>
		public static DataTable GetComponentOrders()
		{
			DataTable dtOrders = DAL.OrderedComponent.GetComponentOrders();
			return dtOrders;
		}

		///<Developers>
		///	<Developer>Carl Jensen</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>6/16/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="2930"> 
		///		<ExpectedInput>Valid division code</ExpectedInput>
		///		<ExpectedOutput>DataTable of all valid component classes</ExpectedOutput>
		///	</Case>
		///	
		///<Case type="1" testid ="394"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Returns a data table of OrderableComponentClasses for a division
		/// </summary>
		/// <returns></returns>
		public static DataTable GetOrderableComponentClasses()
		{
			return DAL.OrderedComponent.GetOrderableComponentClasses();
		}

		///<Developers>
		///	<Developer>Carl Jensen</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>6/16/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="2932"> 
		///		<ExpectedInput>Valid parameters</ExpectedInput>
		///		<ExpectedOutput>true</ExpectedOutput>
		///</Case>
		///
		///<Case type="1" testid ="2933"> 
		///		<ExpectedInput>Invalid parameters</ExpectedInput>
		///		<ExpectedOutput>Exception</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Updates the status of an ordered component, Implements BR_3.03
		/// </summary>
		/// <param name="orderStatusCode"></param>
		/// <param name="taskStatusCode"></param>
		/// <param name="userName"></param>
		/// <param name="cannedCommentGuid"></param>
		/// <param name="cannedCommentText"></param>
		/// <param name="saveSpecimen"></param>
		/// <param name="exRep"></param>
		/// <param name="lastUpdateFunctionId">UC or calling method</param>
		public bool UpdateStatus(char orderStatusCode, 
			char taskStatusCode, 
			string userName, 
			Guid cannedCommentGuid, 
			string cannedCommentText, 
			bool saveSpecimen, 
			BOL.ExceptionReport exRep,
			Common.UpdateFunction lastUpdateFunctionId)
		{
			//Create new Order Component table 
			DataTable dtOrderComponent = InitializeOrderedComponentTable();
			dtOrderComponent.TableName = "dtOrderComponent";

			//Create a new Row in Order Component table and add data based on the information passed in
			DataRow drOrderComponent = dtOrderComponent.NewRow();

			drOrderComponent[TABLE.OrderedComponent.OrderedComponentGuid] = this.OrderedComponentGuid;
			drOrderComponent[TABLE.OrderedComponent.SpecimenAccessionNumber] = this.SpecimenAccessionNumber;
			
			if (this.SpecimenIdentified) // CR3114
			{
				drOrderComponent[TABLE.OrderedComponent.SpecimenWorkloadUid] = GetSpecimen().SpecimenUid;
			}

			if (saveSpecimen && orderStatusCode != Common.Utility.GetOrderStatusCodeFromEnum(Common.OrderStatus.Cancelled))
			{
				drOrderComponent[TABLE.OrderedComponent.PatientSpecimenGuid] = GetSpecimen().SpecimenGuid;
			}
			else
			{
				drOrderComponent[TABLE.OrderedComponent.PatientSpecimenGuid] =  System.DBNull.Value;
			}
			if (cannedCommentGuid == Guid.Empty) 
			{
				drOrderComponent[TABLE.OrderedComponent.CancellationCannedCommentGuid] = System.DBNull.Value;
			}
			else
			{
				drOrderComponent[TABLE.OrderedComponent.CancellationCannedCommentGuid] = cannedCommentGuid;
			}
			drOrderComponent[TABLE.OrderedComponent.OrderStatusCode] = orderStatusCode;
			drOrderComponent[TABLE.OrderedComponent.LastUpdateUser] = userName;
			drOrderComponent[TABLE.OrderedComponent.CancellationText]	= cannedCommentText;
			drOrderComponent[TABLE.OrderedComponent.InappropriateRequestIndicator] = _inappropriateRequestIndicator;
			drOrderComponent[TABLE.OrderedComponent.NoInappropriateReportText] =	_noInappropriateReportText;
			drOrderComponent[TABLE.OrderedComponent.EmergencyOrderIndicator] =	_emergencyOrderIndicator;

			drOrderComponent[TABLE.OrderedComponent.RowVersion] = this.RowVersion;

			//Add Order Component row to the Order Component table
			dtOrderComponent.Rows.Add(drOrderComponent);

			DataTable dtAcceptException = null;
			if (exRep!=null)
			{
				dtAcceptException = exRep.ExceptionData.Table.Clone();
				dtAcceptException.Rows.Add(exRep.ExceptionData.ItemArray);
			}

			System.Guid [] orderGuids = new Guid[1];
			orderGuids[0] = this.OrderedComponentGuid;

			System.Data.DataTable dtWorkload = null;
			//CR 2212
			if(orderStatusCode == 'X')
			{
				dtWorkload = BOL.WorkloadEvent.GenerateWorkloadData(Common.WorkloadProcessID.CancelOrder, System.Guid.Empty, this.OrderedComponentGuid, orderGuids, Common.WorkloadTransactionType.Miscellanious, false);
			}
			else if(orderStatusCode == 'A')
			{
				dtWorkload = BOL.WorkloadEvent.GenerateWorkloadData(Common.WorkloadProcessID.AcceptOrder, System.Guid.Empty, this.OrderedComponentGuid, orderGuids, Common.WorkloadTransactionType.Miscellanious, false);
			}
			dtWorkload = Common.Utility.AppendLastUpdateInformation(dtWorkload,lastUpdateFunctionId);
			if (UpdateCPRS(this.CPRSOrderNumber, orderStatusCode, taskStatusCode, _inappropriateRequestIndicator, cannedCommentGuid, cannedCommentText))
			{
				return DAL.OrderedComponent.UpdateStatus(Common.Utility.AppendLastUpdateInformation(dtOrderComponent,lastUpdateFunctionId), dtAcceptException, dtWorkload);
			}
			else
			{
				return false;
			}
		}

		/// <summary>
		/// UpdateCPRS
		/// </summary>
		/// <param name="cprsOrderNumber"></param>
		/// <param name="orderStatusCode"></param>
		/// <param name="taskStatusCode"></param>
		/// <param name="inappropriateRequestIndicator"></param>
		/// <param name="cannedCommentGuid"></param>
		/// <param name="cannedCommentText"></param>
		/// <returns></returns>
		private bool UpdateCPRS(string cprsOrderNumber,
			char orderStatusCode,
			char taskStatusCode,
			bool inappropriateRequestIndicator,
			Guid cannedCommentGuid,
			string cannedCommentText)
		{
			DataSet ds = DAL.OrderedComponent.GetOrderData(cprsOrderNumber);			

			if (ds.Tables.Count == 1 && ds.Tables[0].Rows.Count == 1)
			{
				DataTable dtOrderData = ds.Tables[0];

				dtOrderData.Rows[0][TABLE.OrderedComponent.OrderStatusCode] = orderStatusCode;
				dtOrderData.Rows[0][TABLE.OrderedComponent.TaskStatusCode] = taskStatusCode;
				dtOrderData.Rows[0][TABLE.OrderedComponent.InappropriateRequestIndicator] = inappropriateRequestIndicator;
				if (cannedCommentGuid == Guid.Empty)
				{
					dtOrderData.Rows[0][TABLE.OrderedComponent.CancellationCannedCommentGuid] = System.DBNull.Value;
				}
				else
				{
					dtOrderData.Rows[0][TABLE.OrderedComponent.CancellationCannedCommentGuid] = cannedCommentGuid;
				}
				dtOrderData.Rows[0][TABLE.CannedComment.CannedCommentText] = cannedCommentText;
			
				new VbecsOmgMessage().PatientOrderUpdate(dtOrderData);
				return true;
			}
			else
			{
				throw new BOL.BusinessObjectException("Data error: More than 1 row returned");
			}
		}

		/// <summary>
		/// This method sends the order completion message to Vista
		/// </summary>
		/// <param name="cprsOrderNumber"></param>
		/// <returns></returns>
		public static bool SendComponentOrderCompletionMessageToVista(string cprsOrderNumber)
		{
			DataSet ds = DAL.OrderedComponent.GetOrderData(cprsOrderNumber);			

			if (ds.Tables.Count == 1 && ds.Tables[0].Rows.Count == 1)
			{
				DataTable dtOrderData = ds.Tables[0];

				dtOrderData.Rows[0][TABLE.OrderedComponent.OrderStatusCode] = Common.Utility.GetTaskStatusCodeFromEnum(Common.TaskStatus.Completed);
				dtOrderData.Rows[0][TABLE.OrderedComponent.TaskStatusCode] = Common.Utility.GetTaskStatusCodeFromEnum(Common.TaskStatus.Completed);
				dtOrderData.Rows[0][TABLE.OrderedComponent.InappropriateRequestIndicator] = false;
				dtOrderData.Rows[0][TABLE.OrderedComponent.CancellationCannedCommentGuid] = System.DBNull.Value;
				dtOrderData.Rows[0][TABLE.CannedComment.CannedCommentText] = "";
			
				new VbecsOmgMessage().PatientOrderUpdate(dtOrderData);
				return true;
			}
			else
			{
				throw new BOL.BusinessObjectException("Data error: More than 1 row returned");
			}
		}


		///<Developers>
		///	<Developer>Rob Heiberger</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/18/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="6395"> 
		///		<ExpectedInput>None</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="6396"> 
		///		<ExpectedInput>None</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// InitializeOrderedComponentTable
		/// </summary>
		/// <returns></returns>
		public static DataTable InitializeOrderedComponentTable()
		{
			DataTable dtComponentOrder = new DataTable();

			dtComponentOrder.Columns.Add(TABLE.OrderedComponent.OrderedComponentGuid, typeof(Guid));
			dtComponentOrder.Columns.Add(TABLE.OrderedComponent.PatientOrderGuid, typeof(Guid));
			dtComponentOrder.Columns.Add(TABLE.OrderedComponent.ComponentClassId, typeof(int));
			dtComponentOrder.Columns.Add(TABLE.OrderedComponent.CprsOrderNumber, typeof(string));
			dtComponentOrder.Columns.Add(TABLE.OrderedComponent.VbecsOrderNumber, typeof(int));
			dtComponentOrder.Columns.Add(TABLE.OrderedComponent.RequiredUnitQuantity, typeof(decimal));
			dtComponentOrder.Columns.Add(TABLE.OrderedComponent.SpecimenRequiredIndicator, typeof(byte));
			dtComponentOrder.Columns.Add(TABLE.OrderedComponent.PatientSpecimenGuid, typeof(Guid));
			dtComponentOrder.Columns.Add(TABLE.OrderedComponent.SpecimenStatusCode, typeof(string));
			dtComponentOrder.Columns.Add(TABLE.OrderedComponent.PreOpIndicator, typeof(byte));
			dtComponentOrder.Columns.Add(TABLE.OrderedComponent.OrderStatusCode, typeof(string));
			dtComponentOrder.Columns.Add(TABLE.OrderedComponent.OrderUrgencyCode, typeof(string));
			dtComponentOrder.Columns.Add(TABLE.OrderedComponent.RequiredDatetime, typeof(DateTime));
			dtComponentOrder.Columns.Add(TABLE.OrderedComponent.OrderSpecialInstructions, typeof(string));
			dtComponentOrder.Columns.Add(TABLE.OrderedComponent.OrderedComponentComments, typeof(string));
			dtComponentOrder.Columns.Add(TABLE.OrderedComponent.InappropriateRequestIndicator, typeof(bool));
			dtComponentOrder.Columns.Add(TABLE.OrderedComponent.TaskStatusCode, typeof(string));
			dtComponentOrder.Columns.Add(TABLE.OrderedComponent.CancellationCannedCommentGuid, typeof(Guid));
			dtComponentOrder.Columns.Add(TABLE.OrderedComponent.CancellationText, typeof(string));
			dtComponentOrder.Columns.Add(TABLE.OrderedComponent.DivisionCode, typeof(string));
			dtComponentOrder.Columns.Add(TABLE.OrderedComponent.LastUpdateDate, typeof(DateTime));
			dtComponentOrder.Columns.Add(TABLE.OrderedComponent.LastUpdateUser, typeof(string));
			dtComponentOrder.Columns.Add(TABLE.OrderedComponent.LabTestId, typeof(string));
			dtComponentOrder.Columns.Add(TABLE.OrderedComponent.SpecimenAccessionNumber, typeof(string));
			dtComponentOrder.Columns.Add(TABLE.OrderedComponent.SpecimenWorkloadUid, typeof(string));
			dtComponentOrder.Columns.Add(TABLE.OrderedComponent.NoInappropriateReportText, typeof(string));
			dtComponentOrder.Columns.Add(TABLE.OrderedComponent.EmergencyOrderIndicator, typeof(bool));
			dtComponentOrder.Columns.Add(TABLE.OrderedComponent.RowVersion, typeof(byte[]));
			dtComponentOrder.Columns.Add(TABLE.OrderedComponent.LastUpdateFunctionId, typeof(int));

			return dtComponentOrder;
		}

		
		///<Developers>
		///	<Developer>Rob Heiberger</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/18/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="6400"> 
		///		<ExpectedInput>Negative ABS, Anti-C from VBECS</ExpectedInput>
		///		<ExpectedOutput>false</ExpectedOutput>
		///	</Case>
		///
		///<Case type="0" testid ="9143"> 
		///		<ExpectedInput>Negative ABS, Anti-C from conversion</ExpectedInput>
		///		<ExpectedOutput>false</ExpectedOutput>
		///	</Case>
		///
		///<Case type="0" testid ="9144"> 
		///		<ExpectedInput>Positive ABS, no TRs</ExpectedInput>
		///		<ExpectedOutput>false</ExpectedOutput>
		///	</Case>
		///
		///<Case type="0" testid ="9145"> 
		///		<ExpectedInput>Negative ABS, no TRs</ExpectedInput>
		///		<ExpectedOutput>true</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="6401"> 
		///		<ExpectedInput>Invalid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Checks if the test is eligible for IS only crossmatch - BR_40.05
		/// </summary>
		/// <returns></returns>
		public bool CheckIfISOnlyXMatchPossible()
		{
			bool PositiveABS = false;
			bool AntigenNegativeReqExists = false;
			string ABSInterpBloodTestType = "";
			string TASABSInterpBloodTestType = "";
			int ABSInterpCount = 0;

			DataTable dtTestResults = this.GetSpecimen().GetTestResultsForSpecimen(); // CR3114
			DataTable dtTransfusionRequirements = BOL.PatientTransfusionRequirement.GetTransfusionRequirementsForPatient(this.Patient.PatientGuid);
			DataTable dtBloodTestTypes = BOL.BloodTestType.GetBloodTestTypes(((int)Common.OrderableTest.ABS).ToString());
			
			if (dtBloodTestTypes.Select(TABLE.BloodTestType.BloodTestName + " LIKE '*Interp*'").Length > 0)
			{
				ABSInterpBloodTestType = dtBloodTestTypes.Select(TABLE.BloodTestType.BloodTestName + " LIKE '*Interp*'")[0][TABLE.BloodTestType.BloodTestTypeId].ToString().Trim();
			}

			dtBloodTestTypes = BOL.BloodTestType.GetBloodTestTypes(((int)Common.OrderableTest.TAS).ToString());
			if (dtBloodTestTypes.Select(TABLE.BloodTestType.BloodTestName + " LIKE '*Antibody Screen Interp*'").Length > 0)
			{
				TASABSInterpBloodTestType = dtBloodTestTypes.Select(TABLE.BloodTestType.BloodTestName + " LIKE '*Antibody Screen Interp*'")[0][TABLE.BloodTestType.BloodTestTypeId].ToString().Trim();
			}

			//checking if there are any positive results of ABS testing
			foreach (DataRow drTestResult in dtTestResults.Rows)
			{
				if (drTestResult[TABLE.BloodTestType.BloodTestTypeId].ToString().Trim() == ABSInterpBloodTestType ||
					drTestResult[TABLE.BloodTestType.BloodTestTypeId].ToString().Trim() == TASABSInterpBloodTestType)
				{
					ABSInterpCount ++;
					if (drTestResult[TABLE.TestResult.TestResultId].ToString().Trim() != Common.TestResult.N.ToString())
					{
						PositiveABS = true;
					}
					else
					{
						PositiveABS = false;
					}
				}
			}

			if (ABSInterpCount == 0)
				PositiveABS = true;

			//CR 2900 added check for converted transfusion requirements (RecordStatusCode = C)
			if (dtTransfusionRequirements.Select(TABLE.PatientTransfusionRequirement.TransfusionRequirementCategoryCode + " = '" + Common.Utility.GetTransfusionCategoryCodeFromEnum(Common.TransfusionRequirementCategoryCode.AntigenNegative) + "' AND " + TABLE.PatientTransfusionRequirement.RecordStatusCode + " IN ('A','C')").Length > 0)
			{
				AntigenNegativeReqExists = true;
			}
			else if (dtTransfusionRequirements.Select(TABLE.PatientTransfusionRequirement.TransfusionRequirementCategoryCode + " = '" + Common.Utility.GetTransfusionCategoryCodeFromEnum(Common.TransfusionRequirementCategoryCode.AntibodyIdentified) + "' AND " + TABLE.PatientTransfusionRequirement.RecordStatusCode + " IN ('A','C')").Length > 0)
			{
				DataTable dtAntibodies = BOL.AntibodyType.GetAntibodyTypeLookupList();
				DataRow[] drTRs = dtTransfusionRequirements.Select(TABLE.PatientTransfusionRequirement.TransfusionRequirementCategoryCode + " = '" + Common.Utility.GetTransfusionCategoryCodeFromEnum(Common.TransfusionRequirementCategoryCode.AntibodyIdentified) + "' AND " + TABLE.PatientTransfusionRequirement.RecordStatusCode + " IN ('A','C')");
				foreach(DataRow drTR in drTRs)
				{
					if (drTR.IsNull(TABLE.PatientTransfusionRequirement.AntibodyTypeId) == false)
					{
						if (dtAntibodies.Select(TABLE.AntibodyType.AntibodyTypeId + " = " + (int)drTR[TABLE.PatientTransfusionRequirement.AntibodyTypeId] + " AND " + TABLE.AntibodyParameter.HigherLevelOverrideIndicator + " = 1").Length > 0)
						{
							AntigenNegativeReqExists = true;
						}
					}
				}
			}

			if (PositiveABS == true || AntigenNegativeReqExists == true)
			{
				return false;
			}
			else
			{
				return true;
			}
		} 

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>5/24/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="4469"> 
		///		<ExpectedInput>Valid ordered component and blood unit unique identifiers</ExpectedInput>
		///		<ExpectedOutput>Array list containing blood unit test results</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4470"> 
		///		<ExpectedInput>Invalid ordered component and blood unit unique indentifiers</ExpectedInput>
		///		<ExpectedOutput>Argument exception thrown</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Static function to retrieve a list of blood unit test results for a given ordered component (Repeat XMs only)
		/// </summary>
		/// <param name="orderedComponentGuid">Ordered component unique identifier</param>
		/// <param name="bloodUnitGuid">Blood unit unique identifier</param>
		/// <returns>Blood unit test object array list</returns>
		public static ArrayList GetBloodUnitTestResultsForOrderedComponent(Guid orderedComponentGuid, Guid bloodUnitGuid)
		{
			if (orderedComponentGuid == Guid.Empty)
			{
				throw new ArgumentException(Common.StrRes.SysErrMsg.Common.InvalidFormat("ordered component unique identifier").ResString);
			}
			//
			if (bloodUnitGuid == Guid.Empty)
			{
				throw new ArgumentException(Common.StrRes.SysErrMsg.Common.InvalidFormat("blood unit unique identifier").ResString);
			}
			//
			System.Data.DataTable dt = DAL.OrderedComponent.GetBloodUnitTestResultsForOrderedComponent(orderedComponentGuid, bloodUnitGuid);
			//
			System.Collections.ArrayList returnList = new System.Collections.ArrayList();
			foreach(DataRow dr in dt.Rows)
			{
				BOL.BloodUnitTest test = new BOL.BloodUnitTest(dr);
				returnList.Add(test);
			}
			//
			return returnList;
		}

		///<Developers>
		///	<Developer>David Askew</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>5/4/2004</CreationDate>
		///<TestCases>
		///
		///<Case type="0" testid ="4295"> 
		///		<ExpectedInput>DivisionCode, Hours, Date</ExpectedInput>
		///		<ExpectedOutput>DataTable</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4296"> 
		///		<ExpectedInput>Invalid parameters</ExpectedInput>
		///		<ExpectedOutput>Empty DataTable</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// GetRequiredOrders
		/// </summary>
		/// <param name="divisionCode"></param>
		/// <param name="hoursNeeded"></param>
		/// <param name="lastChecked"></param>
		/// <returns>DataTable</returns>
		public static DataTable GetRequiredOrders(string divisionCode, int hoursNeeded, System.DateTime lastChecked)
		{
			return DAL.OrderedComponent.GetRequiredOrders(divisionCode, hoursNeeded, lastChecked);
		}

		///<Developers>
		///	<Developer>David Askew</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>5/4/2004</CreationDate>
		///<TestCases>
		///
		///<Case type="0" testid ="4297"> 
		///		<ExpectedInput>DivisionCode, Date</ExpectedInput>
		///		<ExpectedOutput>DataTable</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4298"> 
		///		<ExpectedInput>Invalid parameters</ExpectedInput>
		///		<ExpectedOutput>Empty DataTable</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// GetOrdersWithSpecimenInBloodBank
		/// </summary>
		/// <param name="divisionCode"></param>
		/// <param name="lastChecked"></param>
		/// <returns>DataTable</returns>
		public static DataTable GetOrdersWithSpecimenInBloodBank(string divisionCode, System.DateTime lastChecked)
		{	
			return DAL.OrderedComponent.GetOrdersWithSpecimenInBloodBank(divisionCode, lastChecked);
		}

		///<Developers>
		///	<Developer>David Askew</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>5/4/2004</CreationDate>
		///<TestCases>
		///
		///<Case type="0" testid ="4299"> 
		///		<ExpectedInput>DivisionCode, Date</ExpectedInput>
		///		<ExpectedOutput>DataTable</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4300"> 
		///		<ExpectedInput>Invalid parameters</ExpectedInput>
		///		<ExpectedOutput>Empty DataTable</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// GetNewOrders
		/// </summary>
		/// <param name="divisionCode"></param>
		/// <param name="lastChecked"></param>
		/// <returns>DataTable</returns>
		public static DataTable GetNewOrders(string divisionCode, System.DateTime lastChecked)
		{	
			return DAL.OrderedComponent.GetNewOrders(divisionCode, lastChecked);
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>5/12/2004</CreationDate>
		///<TestCases>
		///
		///<Case type="0" testid ="6402"> 
		///		<ExpectedInput>Valid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="6403"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Static function, returns a data table containing recently added ordered components
		/// that the user should be alerted about
		/// Implements BR_84.01, BR_84.02
		/// </summary>
		/// <param name="lastChecked">Date and time the user last checked for orders</param>
		/// <returns>Data table of ordered component information</returns>
		public static DataTable GetOrderAlerts(System.DateTime lastChecked)
		{
			DataTable orderAlerts;
			BOL.Division division = new BOL.Division(Common.LogonUser.LogonUserDivisionCode);
			//
			if (division.DivisionAlerts.IconOrderAlertType == Common.OrderAlertTypeCode.All_orders)
			{
				orderAlerts = BOL.OrderedComponent.GetNewOrders(Common.LogonUser.LogonUserDivisionCode, lastChecked);
			}
			else if (division.DivisionAlerts.IconOrderAlertType == Common.OrderAlertTypeCode.Hour_limit)
			{
				orderAlerts = BOL.OrderedComponent.GetRequiredOrders(Common.LogonUser.LogonUserDivisionCode, division.DivisionAlerts.IconOrdersNeededHours, lastChecked);
			}
			else
			{
				orderAlerts = BOL.OrderedComponent.GetOrdersWithSpecimenInBloodBank(Common.LogonUser.LogonUserDivisionCode, lastChecked);
			}
			return orderAlerts;
		}

		///<Developers>
		///	<Developer>Rob Heiberger</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/19/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="6414"> 
		///		<ExpectedInput>Valid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="6415"> 
		///		<ExpectedInput>Invalid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// GetRecentComponentOrders
		/// </summary>
		/// <param name="patientGuid"></param>
		/// <returns></returns>
		public static DataSet GetRecentComponentOrders(System.Guid patientGuid)
		{
			if (patientGuid == Guid.Empty)
			{
				throw new ArgumentException(Common.StrRes.SysErrMsg.Common.InvalidFormat("patient unique identifier").ResString);
			}
			return DAL.OrderedComponent.GetRecentComponentOrders(patientGuid);
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>9/9/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="8193"> 
		///		<ExpectedInput>Valid patient guid</ExpectedInput>
		///		<ExpectedOutput>Count of the recent component orders</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="8194"> 
		///		<ExpectedInput>Invalid patient Guid</ExpectedInput>
		///		<ExpectedOutput>ArgumentException thrown</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Get a count of the recent component orders for a patient
		/// </summary>
		/// <param name="patientGuid">Patient Guid</param>
		/// <returns>Count of the recent component orders</returns>
		public static int GetRecentComponentOrdersCount(System.Guid patientGuid)
		{
			if (patientGuid == Guid.Empty)
			{
				throw new ArgumentException(Common.StrRes.SysErrMsg.Common.InvalidFormat("patient unique identifier").ResString);
			}

			return DAL.OrderedComponent.GetRecentComponentOrdersCount(patientGuid);
		}

		///<Developers>
		///	<Developer>Rob Heiberger</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/19/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="6416"> 
		///		<ExpectedInput>Valid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="6417"> 
		///		<ExpectedInput>Invalid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Get the status of ordered component
		/// </summary>
		/// <param name="orderedComponentGuid"></param>
		/// <returns></returns>
		public static Common.TaskStatus GetStatus(Guid orderedComponentGuid)
		{
			int orderedUnits = DAL.OrderedComponent.GetOrderedComponentQuantity(orderedComponentGuid);
			DataTable dtSelectedUnits = BOL.OrderedUnit.GetOrderedUnitByComponent(orderedComponentGuid, Common.LogonUser.LogonUserDivisionCode);
			int transfusedAndIssuedUnits = dtSelectedUnits.Select(TABLE.UnitStatus.UnitStatusCode + " = 'T' OR " + TABLE.UnitStatus.UnitStatusCode + " = 'I'").Length;
			//CR 2278 Removed for loop that calculated the number of units eligible for issue.
				//also removed the criteria of units ready for issue in the logic below 
				//for determining when to set the status to filled.

			if (transfusedAndIssuedUnits >= orderedUnits)
			{
				return Common.TaskStatus.Filled;
			}
			else
			{
				if (dtSelectedUnits.Rows.Count > 0)
				{
					return Common.TaskStatus.PartiallyCompleted;
				}
				else
				{
					return Common.TaskStatus.NotStarted;
				}
			}
		}

		///<Developers>
		///	<Developer>Krzysztof Dobranowski</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>11/1/2007</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="8514"> 
		///		<ExpectedInput>Not Strated Component Order</ExpectedInput>
		///		<ExpectedOutput>Status "Not Started"</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="0" testid ="8516"> 
		///		<ExpectedInput>Partially Completed Component Order</ExpectedInput>
		///		<ExpectedOutput>Status "Partially Completed"</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="0" testid ="8517"> 
		///		<ExpectedInput>Filled Component Order</ExpectedInput>
		///		<ExpectedOutput>Status "Filled"</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="8515"> 
		///		<ExpectedInput>Invalid component order</ExpectedInput>
		///		<ExpectedOutput>Status "Not Started"</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Get the status of component order and count units eligible for issue (this method is used to decide whether the order completion
		/// message should be sent to CPRS). Implements BR_56.31. CR 2300.
		/// </summary>
		/// <param name="orderedComponentGuid"></param>
		/// <returns></returns>
		public static Common.TaskStatus GetStatusIncludingUnitsEligibleForIssue(Guid orderedComponentGuid)
		{
			int orderedUnits = DAL.OrderedComponent.GetOrderedComponentQuantity(orderedComponentGuid);
			DataTable dtSelectedUnits = BOL.OrderedUnit.GetOrderedUnitByComponent(orderedComponentGuid, Common.LogonUser.LogonUserDivisionCode);
			int transfusedAndIssuedUnits = dtSelectedUnits.Select(TABLE.UnitStatus.UnitStatusCode + " = 'T' OR " + TABLE.UnitStatus.UnitStatusCode + " = 'I'").Length;
			int unitsReadyForIssue = 0;
			ArrayList bloodUnitGuids = new ArrayList();

			for (int i=0; i< dtSelectedUnits.Rows.Count; i++)
			{
				if (Convert.ToChar(dtSelectedUnits.Rows[i][Common.VbecsTables.BloodUnitStatusCodeCurrent.UnitStatusCode]) == Common.Utility.GetUnitStatusCodeCharFromEnum(Common.UnitStatusCode.Assigned) ||
					Convert.ToChar(dtSelectedUnits.Rows[i][Common.VbecsTables.BloodUnitStatusCodeCurrent.UnitStatusCode]) == Common.Utility.GetUnitStatusCodeCharFromEnum(Common.UnitStatusCode.CrossmatchedAssigned))
				{
					//2562
					bloodUnitGuids.Add((Guid)dtSelectedUnits.Rows[i][Common.VbecsTables.BloodUnit.BloodUnitGuid]);
					//2562 end
				}
			}


			// 2562
			if (dtSelectedUnits.Rows.Count > 0)
			{
				unitsReadyForIssue = BOL.PatientUnitIssue.UnitsAvailableForLevelThree(new BOL.Patient((Guid)dtSelectedUnits.Rows[0][Common.VbecsTables.Patient.PatientGuid]), bloodUnitGuids);
			}
			// 2562 end

			if (transfusedAndIssuedUnits + unitsReadyForIssue >= orderedUnits)
			{
				return Common.TaskStatus.Filled;
			}
			else
			{
				if (dtSelectedUnits.Rows.Count > 0)
				{
					return Common.TaskStatus.PartiallyCompleted;
				}
				else
				{
					return Common.TaskStatus.NotStarted;
				}
			}
		}

		#endregion
	}
}
